using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class is a base for all VistALink security 
	/// partial success response messages. 
	/// </summary>
	public abstract class SecurityBasePartialSuccessResponseMessage : SecurityBaseCommentedResponseMessage
	{
		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "secPartialSuccessResponse.xsd";

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_PARTIAL_SUCCESS_NODE_NAME = "PartialSuccessData";

		/// <summary>
		/// This constructor allows creating partial success security response
		/// message from response type. 
		/// </summary>
		/// <param name="responseType">VistALink message response type.</param>
		public SecurityBasePartialSuccessResponseMessage( string responseType )
			: this( responseType, (string)null ) {}

		/// <summary>
		/// This constructor allows creating partial success security response
		/// message from response type and optional informative text message. 
		/// </summary>
		/// <param name="responseType">VistALink message response type.</param>
		/// <param name="message">Human-readable informative text message. Optional - may be null</param>
		public SecurityBasePartialSuccessResponseMessage( string responseType, string message )
			: base( XsdName, responseType, ResponseStatus.PartialSuccess, message ) {}

		/// <summary>
		/// Constructor extending standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="responseType">VistALink message response type.</param>		
		/// <param name="sourceDoc">Source XML document.</param>
		public SecurityBasePartialSuccessResponseMessage( string responseType, XmlDocument sourceDoc )
			: base( XsdName, responseType, sourceDoc )
		{
			if( _responseStatus != ResponseStatus.PartialSuccess )
				throw( new XmlParseException( SR.Exceptions.PartialSuccessInvalidResponseStatus( _responseStatus.ToString() ) ) );
		}

		///<summary>
		/// Overriden method adding content to response element 
		/// of XML message representation during XML serialization. 
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		protected override void WriteXmlResponseNodeContent( XmlWriter writer )
		{
			if( writer == null ) 
				throw( new ArgumentNullException( "writer" ) );			

			writer.WriteStartElement( XMLCONSTS_PARTIAL_SUCCESS_NODE_NAME );
			WriteXmlPartialSuccessAttributes( writer );

			WriteXmlPartialSuccessNodeContent( writer );
				
			writer.WriteEndElement();

			base.WriteXmlResponseNodeContent( writer );
		}
		
		/// <summary>
		/// Derived classes should override this method to add attributes to partial success element 
		/// of XML message representation during XML serialization. 
		/// </summary>
		/// <param name="writer">XmlWriter to use</param>
		protected virtual void WriteXmlPartialSuccessAttributes( XmlWriter writer ) {}

		/// <summary>
		/// Derived classes should override this method to add content to partial success element
		/// of XML message representation during XML serialization. 
		/// </summary>
		/// <param name="writer">XmlWriter to use</param>
		protected virtual void WriteXmlPartialSuccessNodeContent( XmlWriter writer ) {}		

		/// <summary>
		/// XML parsing utility method. Returns reference to partial success data element in 
		/// the specified XML document containing XML representation of VistALink 
		/// partial success response message.
		/// 
		/// If element not found or more than one instance was found - 
		/// XmlParseException exception will be thrown.
		/// </summary>
		/// <param name="sourceDoc">Source XML document.</param>
		/// <returns>Reference to partial success data element.</returns>
		public static XmlElement XmlParseGetPartialSuccessElement( XmlDocument sourceDoc )
		{
			return XmlUtility.ParseGetRequiredElementByUniqueTagName( XmlParseGetResponseElement( sourceDoc ), XMLCONSTS_PARTIAL_SUCCESS_NODE_NAME );
		}
	}
}
